#include <msp430.h>
#include "adf4106.h"
#include "adf4252.h"
#include "adf4360.h"
#include "ad8370.h"
#include "main.h"

/*
 * 10 GHz LO programmer and sequencer
 */

/*
 * waitms
 *
 * delay loop, calibrated to about 1 ms
 * cpu clock ~700 kHz, minimum count is ~700 cycles 
 */
void waitms(int ms) {
	volatile int i;
	volatile int ms_cd = ms;

	while(ms_cd-- > 0) {
		for (i = MS_DELAY_VALUE; i > 0; i--);
	}
}

/*
 * hw_init
 *
 * initializes I/Os to a safe state after power up
 */
void hw_init(void) {
	/* stop watchdog timer */
	WDTCTL = WDTPW | WDTHOLD;

	/* make the PTT pin an input */
	PTT_PDIR &= ~PTT;

	/* disable voltage for RX */
	RX_EN_POUT &= ~RX_EN;
	RX_EN_PDIR |= RX_EN;

	/* disable voltage for TX */
	TX_EN_POUT &= ~TX_EN;
	TX_EN_PDIR |= TX_EN;

	/* set relay pins output, switch to safe position */	
	RLY_IF_POUT &= ~RLY_IF; //TX
	RLY_IF_PDIR |= RLY_IF;

	RLY_RF_POUT &= ~RLY_RF; // RX
	RLY_RF_PDIR |= RLY_RF;

    waitms(200);
	
	adf4360_config();
	adf4252_config();
	adf4106_config();
	ad8370_config();
}

int main(void) {
	enum {SEQ_INIT, SEQ_RX, SEQ_TX};
	int sequencer_state = SEQ_INIT;

	hw_init();
	sequencer_state = SEQ_INIT;
	while(1) {
		/* transition to RX */
		if ((sequencer_state != SEQ_RX) && (PTT_PIN & PTT)) {
			TX_EN_POUT &= ~TX_EN;
			waitms(10);
			RLY_RF_POUT &= ~RLY_RF;
			RLY_IF_POUT |= RLY_IF;
			waitms(10);
			RX_EN_POUT |= RX_EN;
			adf4360_set_freq(1);
			adf4106_set_freq(1);
			sequencer_state = SEQ_RX;
		}
	
		/* transition to TX */
		if ((sequencer_state != SEQ_TX) && !(PTT_PIN & PTT)) {
			RLY_RF_POUT |= RLY_RF;
			RLY_IF_POUT &= ~RLY_IF;
			RX_EN_POUT &= ~RX_EN;
			TX_EN_POUT |= TX_EN;
			adf4360_set_freq(0);
			adf4106_set_freq(0);
			sequencer_state = SEQ_TX;
		}
	}
}
